﻿#pragma once
#include "Component.h"
#include "ConstantBufferLayouts.h"
#include "Align16.h"

namespace RTCam {

class Film;
class Lens;
class Aperture;
class Shutter;
class Texture;

class Camera : public Component, public Align16
{
public:
	Camera(const shared_ptr<Entity>& entity);
	~Camera();

	// Used to animate changes in camera settings
	virtual void FixedUpdate(float curTime, float delta) override;

	void UpdateConstantBuffer(DirectX::XMINT2 renderSize, float curTime, float delta, _Out_ XMMATRIX& view, _Out_ XMMATRIX& proj, _Out_ XMMATRIX& viewProj);

	// Lens movement for focusing the image
	void MoveLensForward();
	void MoveLensBack();
	void MoveFocusForward();
	void MoveFocusBack();

	// Change the aperture size
	void DecreaseFNumber(); // Makes the aperture bigger
	void IncreaseFNumber(); // Makes the aperture smaller
	
	// Change the exposure duration
	void IncreaseShutterSpeed();
	void DecreaseShutterSpeed();

	// Change the film sensitivity
	void IncreaseFilmSpeed();
	void DecreaseFilmSpeed();

	// Change the focal length of the lens
	void IncreaseLensFocalLength();
	void DecreaseLensFocalLength();

	float CalculateDiagonalAngleOfView();
	float CalculateVerticalAngleOfView(float aspectHByW);

	CameraCBuffer m_cameraCBufferData;

	unique_ptr<Film> m_film;
	unique_ptr<Lens> m_lens;
	unique_ptr<Aperture> m_aperture;
	unique_ptr<Shutter> m_shutter;

	string m_currentLensName;
	string m_currentFilmName;

private:
	float CalculateAngleOfView(float lengthAlongDimension);
	float CalculateExposureModifier();
	float CalculateGrainStrength();

	DirectX::XMFLOAT2 CalculateFilmSize(float aspectHByW);

	void UpdateViewMatrix(_Out_ XMMATRIX& view);
	void UpdateLensParameters(DirectX::XMINT2 renderSize, _In_ const XMMATRIX& view, _Out_ XMMATRIX& projection, _Out_ XMMATRIX& viewProjection);
	void UpdateExposureParameters(float curTime);

	// Parameters for animating changes in camera settings
	struct AnimParams {
		float TimeLeft;
		float TotalTime;
		float InitialVal;
		float FinalVal;

		AnimParams();
	};

	void AnimateCameraParam(AnimParams &params, float initialValue, float newValue);

	AnimParams m_fNumberAnim;
	AnimParams m_lensDistanceAnim;
	AnimParams m_focalLengthAnim;
};


}


